//----------------------------------------------------------------------------//
//- CaoDatenBank.pas ---------------------------------------------------------//
//----------------------------------------------------------------------------//

//- Beschreibung -------------------------------------------------------------//
// Die Klasse TCaoDatenbank kapselt alle Datenbankzugriffe                    //
//----------------------------------------------------------------------------//

unit CaoDatenBank;

//------------------------------------------------------------------------------

interface

//------------------------------------------------------------------------------

uses
  SysUtils, Windows, Messages, Classes, Graphics, Controls,
  Forms, Dialogs, CaoPersonenListe, CaoArtikelListe, CaoBoersenInfo;

//------------------------------------------------------------------------------

type
  TCaoDatenbank = class (TObject)

  //----------------------------------------------------------------------------

  public
    procedure SavePersonenListe (obPersonenListe     : TCaoPersonenListe);
    procedure LoadPersonenListe (out obPersonenListe : TCaoPersonenListe);
    procedure SaveArtikelListe  (obArtikelListe      : TCaoArtikelListe);
    procedure LoadArtikelListe  (out obArtikelLIste  : TCaoArtikelListe;
                                 const iBoersenNummer: Integer);
    procedure SaveBoersenInfo   (obBoersenInfo       : TCaoBoersenInfo);
    procedure LoadBoersenInfo   (out obBoersenInfo   : TCaoBoersenInfo);
    procedure LoadAnbieterData  (var obArtikelListe  : TCaoArtikelListe;
                                 var obPersonenListe : TCaoPersonenListe);

  //----------------------------------------------------------------------------

  private
    procedure PrvSetPersonInDb      (obPerson     : TCaoPersonen);
    procedure PrvSetArtikelInDb     (obArtikel    : TCaoArtikel);
    procedure PrvSetAnbieterInDb    (obPerson     : TCaoPersonen;
                                     obArtikel    : TCaoArtikel);
    procedure PrvSetBoersenInfoInDb (obBoersenInfo: TCaoBoersenInfo);

  //----------------------------------------------------------------------------

  end; // TCaoDatenbank

//------------------------------------------------------------------------------

implementation

//------------------------------------------------------------------------------

uses
  CabDmo, CamConst;

//------------------------------------------------------------------------------

procedure TCaoDatenbank.LoadPersonenListe(out obPersonenListe: TCaoPersonenListe);
(*
  Alle Personendaten aus der Datenbank lesen und in das obPersonenListe-Objekt
  schreiben.
*)
var
  iIndex  : Integer;
  obPerson: TCaoPersonen;
begin
  obPersonenListe.FreeAllItems;
  with CaObDatenModul.m_TblPersonen do begin
    Open;
    First;
    for iIndex := 0 to RecordCount -1 do begin
      obPerson := obPersonenListe.CreatePerson;
      obPerson.iPersonenNummer := FieldByName(c_sPersonenPersonenNummer).AsInteger;
      obPerson.sInitialen      := FieldByName(c_sPersonenInitialen)   .AsString;
      obPerson.sName           := FieldByName(c_sPersonenName)        .AsString;
      obPerson.sOrt            := FieldByName(c_sPersonenOrt)         .AsString;
      obPerson.sPlz            := FieldByName(c_sPersonenPlz)         .AsString;
      obPerson.sStrasse        := FieldByName(c_sPersonenStrasse)     .AsString;
      obPerson.sVorname        := FieldByName(c_sPersonenVorname)     .AsString;
      obPerson.iStatus         := c_iModusDefault;
      Next;
    end; // for
    Close;
  end; // with
end; // TCaoDatenbank.LoadPersonenListe

//------------------------------------------------------------------------------

procedure TCaoDatenbank.SavePersonenListe(obPersonenListe: TCaoPersonenListe);
(*
  Je nach Status speichert oder lscht diese Funktion Anbieter in der DB
*)
var
  iIndex: Integer;
begin
  with CaObDatenModul.m_TblPersonen do begin
    Open;
    for iIndex := 0 to obPersonenListe.Count - 1 do begin
      // Speichern der neuen Eintrge
      case obPersonenListe.Items[iIndex].iStatus of

        // Speichern eines Anbieters
        c_iModusNeu, c_iModusCopy:
        begin
          Append;
          PrvSetPersonInDb(obPersonenListe.Items[iIndex]);
          Post;
        end; // case c_iModusNeu

        // Speichern der nderungen an einer Person in der Datenbank
        c_iModusEdit:
        begin
          if Locate(c_sPersonenPersonenNummer,
                    obPersonenListe.Items[iIndex].iPersonenNummer, []) then begin
            Edit;
            PrvSetPersonInDb(obPersonenListe.Items[iIndex]);
            Post;
          end; // if
        end; // case c_iModusEdit

        // Lschen einer Person aus der Datenbank
        c_iModusDelete:
        begin
          if Locate(c_sPersonenPersonenNummer,
                    obPersonenListe.Items[iIndex].iPersonenNummer, []) then begin
            Delete;
          end; // if
        end; // case c_iModusDelete

      end; // case
      obPersonenListe.Items[iIndex].iStatus := c_iModusDefault;
    end; // for
    Close;
  end; // with
end; // TCaoDatenbank.SavePersonenListe

//------------------------------------------------------------------------------

procedure TCaoDatenbank.SaveArtikelListe(obArtikelListe: TCaoArtikelListe);
(*
  nderungen in der Artikelliste in die Datenbank schreiben
*)
var
  iIndex: Integer;
begin
  with CaObDatenModul do begin
    //m_TblArtikel.Open;
    //m_TblAnbieter.Open;
    for iIndex := 0 to obArtikelListe.Count - 1 do begin
      // Speichern der neuen Eintrge
      case obArtikelListe.Items[iIndex].iStatus of

        c_iModusNeu, c_iModusCopy:
        begin
          // Speichern der Artikeldaten
          m_TblArtikel.Open;
          m_TblArtikel.Append;
          PrvSetArtikelInDb(obArtikelListe.Items[iIndex]);
          m_TblArtikel.Post;
          obArtikelListe.Items[iIndex].iArtikelNummer :=
                    m_TblArtikel.FieldByName(c_sArtikelArtikelNummer).AsInteger;
           m_TblArtikel.Close;

          // Speichern der Anbieterdaten
          m_TblAnbieter.Open;
          m_TblAnbieter.Append;
          PrvSetAnbieterInDb(TCaoPersonen(obArtikelListe.Items[iIndex].pAnbieter),
                             obArtikelListe.Items[iIndex]);
          m_TblAnbieter.Post;
          m_TblAnbieter.Close;
        end; // case c_iModusNeu

        // Speichern der nderungen an einem Artikel in der Datenbank
        c_iModusEdit:
        begin
          m_TblArtikel.Open;
          // Den Datenbankpointer auf den genderten Artikel setzen
          if m_TblArtikel.Locate(c_sArtikelArtikelNummer,
                    obArtikelListe.Items[iIndex].iArtikelNummer, []) then begin
            m_TblArtikel.Edit;
            PrvSetArtikelInDb(obArtikelListe.Items[iIndex]);
            m_TblArtikel.Post;
          end; // if
          m_TblArtikel.Close;
        end; // case c_iModusEdit

        // Lschen einer Person aus der Datenbank
        c_iModusDelete:
        begin
          m_TblArtikel.Open;
          m_TblAnbieter.Open;
          // Den Datenbankpointer in der ArtikelListe auf den zu lschenden
          // Artikel setzen
          if m_TblArtikel.Locate(c_sArtikelArtikelNummer,
                    obArtikelListe.Items[iIndex].iArtikelNummer, []) then begin
            m_TblArtikel.Delete;
          end; // if
          // Aus der AnbieterListe muss der Eintrag mit dem gelschten Artikel
          // ebenfalls gelscht werden
          if m_TblAnbieter.Locate(c_sAnbieterArtikelNummer,
                    obArtikelListe.Items[iIndex].iArtikelNummer, []) then begin
            m_TblAnbieter.Delete;
          end; // if
          m_TblArtikel.Close;
          m_TblAnbieter.Close;
        end; // case c_iModusDelete

      end; // case
      obArtikelListe.Items[iIndex].iStatus := c_iModusDefault;
    end; // for
    m_TblArtikel.Close;
    m_TblAnbieter.Close;
  end; // with
end; // TCaoDatenbank.SaveArtikelListe

//------------------------------------------------------------------------------

procedure TCaoDatenbank.PrvSetArtikelInDb(obArtikel: TCaoArtikel);
(*
  Artikeldaten in die Datenbank schreiben
*)
begin
  with CaObDatenModul.m_TblArtikel do begin
    FieldByName(c_sArtikelBeschreibung) .AsString  := obArtikel.sBeschreibung;
    FieldByName(c_sArtikelBoersenNummer).AsInteger := obArtikel.iBoersenNummer;
    FieldByName(c_sArtikelPreis)        .AsFloat   := obArtikel.dPreis;
    FieldByName(c_sArtikelVerkauft)     .AsBoolean := obArtikel.fVerkauft;
  end; // with
end; // TCaoDatenbank.PrvSetArtikelInDb

//------------------------------------------------------------------------------

procedure TCaoDatenbank.SaveBoersenInfo(obBoersenInfo: TCaoBoersenInfo);
(*
  Speichern der Brsendaten
*)
begin
  with CaObDatenModul.m_TblBoersenInfo do begin
    case obBoersenInfo.iStatus of
      c_iModusNeu:  begin
                      Open;
                      Append;
                      PrvSetBoersenInfoInDb(obBoersenInfo);
                      Post;
                      obBoersenInfo.iBoersenNummer :=
                        FieldByName(c_sBoersenBoersenNummer).AsInteger;
                      Close;
                    end; // case c_iModusNeu
      c_iModusEdit: begin
                      Open;
                      // berprfen, ob der Eintrag existiert
                      if Locate(c_sBoersenBoersenNummer,
                         obBoersenInfo.iBoersenNummer, []) then begin
                        Edit;
                        PrvSetBoersenInfoInDb(obBoersenInfo);
                        Post;
                      end; // if
                      Close;
                    end; // case c_iModusEdit
    end; // case
    obBoersenInfo.iStatus := c_iModusDefault;
  end; // with
end; // TCaoDatenbank.SaveBoersenInfo

//------------------------------------------------------------------------------

procedure TCaoDatenbank.PrvSetPersonInDb(obPerson: TCaoPersonen);
(*
  Speichert Personendaten in den Datenbankfeldern
*)
begin
  with CaObDatenModul.m_TblPersonen do begin
    FieldByName(c_sPersonenInitialen).AsString       := obPerson.sInitialen;
    FieldByName(c_sPersonenName).AsString            := obPerson.sName;
    FieldByName(c_sPersonenOrt).AsString             := obPerson.sOrt;
    FieldByName(c_sPersonenPersonenNummer).AsInteger := obPerson.iPersonenNummer;
    FieldByName(c_sPersonenPlz).AsString             := obPerson.sPlz;
    FieldByName(c_sPersonenStrasse).AsString         := obPerson.sStrasse;
    FieldByName(c_sPersonenVorname).AsString         := obPerson.sVorname;
  end; // with
end; // TCaoDatenbank.PrvSetPersonInDb

//------------------------------------------------------------------------------

procedure TCaoDatenbank.PrvSetAnbieterInDb(obPerson : TCaoPersonen;
                                           obArtikel: TCaoArtikel);
(*
  Speichert Anbieterdaten in der Datenbank
*)
begin
  with CaObDatenModul.m_TblAnbieter do begin
    FieldByName(c_sAnbieterPersonenNummer).AsInteger := obPerson.iPersonenNummer;
    FieldByName(c_sAnbieterArtikelNummer).AsInteger  := obArtikel.iArtikelNummer;
  end; // with
end; // TCaoDatenbank.PrvSetAnbieterInDb

//------------------------------------------------------------------------------

procedure TCaoDatenbank.LoadArtikelListe(out   obArtikelLIste: TCaoArtikelListe;
                                         const iBoersenNummer: Integer);
(*
  Laden der ArtikelListe aus der Datenbank
*)
var
  iIndex   : Integer;
  obArtikel: TCaoArtikel;
begin
  obArtikelLIste.FreeAllItems;
  with CaObDatenModul.m_TblArtikel do begin
    Open;
    First;
    for iIndex := 0 to RecordCount -1 do begin
      if FieldByName(c_sArtikelBoersenNummer).AsInteger = iBoersenNummer then begin
        obArtikel                := obArtikelListe.CreateArtikel;
        obArtikel.dPreis         := FieldByName(c_sArtikelPreis)        .AsFloat;
        obArtikel.fVerkauft      := FieldByName(c_sArtikelVerkauft)     .AsBoolean;
        obArtikel.iArtikelNummer := FieldByName(c_sArtikelArtikelNummer).AsInteger;
        obArtikel.sBeschreibung  := FieldByName(c_sArtikelBeschreibung) .AsString;
        obArtikel.iBoersenNummer := FieldByName(c_sArtikelBoersenNummer).AsInteger;
        obArtikel.iStatus        := c_iModusDefault;
      end; // if
      Next;
    end; // for
    Close;
  end; // with
end; // TCaoDatenbank.LoadArtikelListe

//------------------------------------------------------------------------------

procedure TCaoDatenbank.LoadAnbieterData(var obArtikelListe : TCaoArtikelListe;
                                         var obPersonenListe: TCaoPersonenListe);
(*
  Zuordnen der Artikel zu den Anbietern
*)
var
  obPerson  : TCaoPersonen;
  obArtikel : TCaoArtikel;
  iIndex    : Integer;
begin
  with CaObDatenModul.m_TblAnbieter do begin
    Open;
    First;
    for iIndex  := 0 to RecordCount -1 do begin
      // Person aus der Personenliste lesen
      obPerson  := obPersonenListe.Locate(FieldByName
                                         (c_sAnbieterPersonenNummer).AsInteger);
      // Artikel aus der Artikelliste lesen
      obArtikel := obArtikelListe.Locate(FieldByName
                                         (c_sAnbieterArtikelNummer).AsInteger);
      if Assigned(obArtikel) then begin
        obArtikel.pAnbieter := Pointer(obPerson);
        // Den Artikel der Artikelliste der Person hinzufgen
        obPerson.obArtikelListe.AddArtikel(obArtikel);
      end; // if
      Next;
    end; // while
    Close;
  end; // with
end; // TCaoDatenbank.LoadAnbieterData

//------------------------------------------------------------------------------

procedure TCaoDatenbank.PrvSetBoersenInfoInDb(obBoersenInfo: TCaoBoersenInfo);
(*
  Brseninformation in die Datenbank schreiben
*)
begin
  with CaObDatenModul.m_TblBoersenInfo do begin
    FieldByName(c_sBoersenAbgabeBetrag).AsFloat     := obBoersenInfo.dAbgabeBetrag;
    FieldByName(c_sBoersenAbgabeProzent).AsFloat    := obBoersenInfo.dAbgabeProzent;
    FieldByName(c_sBoersenTitel).AsString           := obBoersenInfo.sTitel;
    FieldByName(c_sBoersenBoersenDatum).AsDateTime  := obBoersenInfo.dtBoersenDatum;
    FieldByName(c_sBoersenAbschlussDatum).AsDateTime:= obBoersenInfo.dtAbschlussDatum;
    FieldByName(c_sBoerseActive).AsBoolean          := obBoersenInfo.fBoerseActive;
  end; // with
end; // TCaoDatenbank.PrvSetBoersenInfoInDb

//------------------------------------------------------------------------------

procedure TCaoDatenbank.LoadBoersenInfo(out obBoersenInfo: TCaoBoersenInfo);
(*
  Laden der Brsendaten aus der Datenbank. Es wird entweder die aktive
  Brse geladen, oder wenn keine Aktive Brse gefunden wurde, werden
  Defaultwerte zurckgegeben. Der Rckgabewert gibt an, ob eine Brse gefunden
  wurde oder nicht.
*)
var
  fFoundBoersenInfo: Boolean;
begin
  fFoundBoersenInfo := False;
  with CaObDatenModul.m_TblBoersenInfo do begin
    open;
    if RecordCount > 0 then begin
      First;
      repeat
        if FieldByName(c_sBoerseActive).AsBoolean then begin
          obBoersenInfo.dAbgabeBetrag
            := FieldByName(c_sBoersenAbgabeBetrag).AsFloat;
          obBoersenInfo.dAbgabeProzent
            := FieldByName(c_sBoersenAbgabeProzent).AsFloat;
          obBoersenInfo.dtBoersenDatum
            := FieldByName(c_sBoersenBoersenDatum).AsDateTime;
          obBoersenInfo.dtAbschlussDatum
            := FieldByName(c_sBoersenAbschlussDatum).AsDateTime;
          obBoersenInfo.sTitel
            := FieldByName(c_sBoersenTitel).AsString;
          obBoersenInfo.iBoersenNummer
            := FieldByName(c_sBoersenBoersenNummer).AsInteger;
          obBoersenInfo.fBoerseActive
            := FieldByName(c_sBoerseActive).AsBoolean;
          obBoersenInfo.iStatus
            := c_iModusDefault;
          fFoundBoersenInfo := True;
        end; // if
      until not FindNext;
    end; // if
    // Wurde keine Aktive Brse gefunden, so werden Defaultwerte geladen
    if not fFoundBoersenInfo then begin
      obBoersenInfo.dAbgabeBetrag           := 0;
      obBoersenInfo.dAbgabeProzent          := 0;
      obBoersenInfo.dtBoersenDatum          := Now;
      obBoersenInfo.dtAbschlussDatum        := 0;  
      obBoersenInfo.sTitel                  := '';
      obBoersenInfo.iStatus                 := c_iModusNeu;
    end; // if
    close;
  end; // with
end; // TCaoDatenbank.LoadBoersenInfo

//------------------------------------------------------------------------------

end. // CaoDatenBank
